import Foundation

// MARK: - Exercise Frequency Enum
public enum ExerciseFrequency: String, CaseIterable, Codable, Hashable {
    case none = "none"
    case rarely = "rarely" // 1-2 times per month
    case occasional = "occasional" // 1-2 times per week
    case regular = "regular" // 3-4 times per week
    case frequent = "frequent" // 5+ times per week
    
    public var displayName: String {
        switch self {
        case .none: return "No Exercise"
        case .rarely: return "Rarely (1-2/month)"
        case .occasional: return "Occasional (1-2/week)"
        case .regular: return "Regular (3-4/week)"
        case .frequent: return "Frequent (5+/week)"
        }
    }
    
    public var hazardRatio: Double {
        switch self {
        case .none: return 1.0
        case .rarely: return 0.9
        case .occasional: return 0.85
        case .regular: return 0.75
        case .frequent: return 0.65
        }
    }
}

// MARK: - Smoking Status Enum
public enum SmokingStatus: String, CaseIterable, Codable, Hashable {
    case never = "never"
    case former = "former"
    case current = "current"
    
    public var displayName: String {
        switch self {
        case .never: return "Never Smoked"
        case .former: return "Former Smoker"
        case .current: return "Current Smoker"
        }
    }
    
    public var hazardRatio: Double {
        switch self {
        case .never: return 1.0
        case .former: return 1.3
        case .current: return 2.5
        }
    }
}

// MARK: - Vaping Status Enum
enum VapingStatus: String, CaseIterable, Codable, Hashable {
    case never = "never"
    case former = "former"
    case current = "current"
    
    var displayName: String {
        switch self {
        case .never: return "Never Vaped"
        case .former: return "Former Vaper"
        case .current: return "Current Vaper"
        }
    }
}

// MARK: - Survey Data Model
struct SurveyData: Codable, Equatable, Hashable {
    // Smoking information
    var smokingStatus: SmokingStatus = .never
    var cigarettesPerDay: Int = 0
    
    // Vaping information
    var vapingStatus: VapingStatus = .never
    var puffsPerDay: Int = 0
    
    // Alcohol consumption
    var alcoholUnitsPerWeek: Double = 0.0
    
    // Diet information
    var fastFoodMealsPerWeek: Int = 0
    
    // Exercise information
    var exerciseFrequency: ExerciseFrequency = .none
    
    // Survey completion tracking
    var isCompleted: Bool = false
    var lastUpdated: Date = Date()
    
    // MARK: - Validation Methods
    
    var isValid: Bool {
        return isCigarettesValid && 
               isPuffsValid && 
               isAlcoholValid && 
               isFastFoodValid
    }
    
    var isCigarettesValid: Bool {
        guard smokingStatus == .current else { return true }
        return cigarettesPerDay >= 0 && cigarettesPerDay <= 100
    }
    
    var isPuffsValid: Bool {
        guard vapingStatus == .current else { return true }
        return puffsPerDay >= 0 && puffsPerDay <= 1000
    }
    
    var isAlcoholValid: Bool {
        return alcoholUnitsPerWeek >= 0 && alcoholUnitsPerWeek <= 50
    }
    
    var isFastFoodValid: Bool {
        return fastFoodMealsPerWeek >= 0 && fastFoodMealsPerWeek <= 21
    }
    
    // MARK: - Validation Error Messages
    
    func validationErrors() -> [String] {
        var errors: [String] = []
        
        if !isCigarettesValid {
            errors.append("Cigarettes per day must be between 0-100")
        }
        
        if !isPuffsValid {
            errors.append("Vape puffs per day must be between 0-1000")
        }
        
        if !isAlcoholValid {
            errors.append("Alcohol units per week must be between 0-50")
        }
        
        if !isFastFoodValid {
            errors.append("Fast food meals per week must be between 0-21")
        }
        
        return errors
    }
    
    // MARK: - Risk Assessment
    
    var overallRiskLevel: RiskLevel {
        let smokingRisk = smokingStatus == .current && cigarettesPerDay > 0
        let vapingRisk = vapingStatus == .current && puffsPerDay > 50
        let alcoholRisk = alcoholUnitsPerWeek > 14 // UK guidelines
        let dietRisk = fastFoodMealsPerWeek > 4
        let exerciseRisk = exerciseFrequency == .none || exerciseFrequency == .rarely
        
        let riskFactors = [smokingRisk, vapingRisk, alcoholRisk, dietRisk, exerciseRisk].filter { $0 }.count
        
        switch riskFactors {
        case 0: return .low
        case 1...2: return .moderate
        case 3...4: return .high
        default: return .veryHigh
        }
    }
}

// MARK: - Risk Level Enum
enum RiskLevel: String, CaseIterable {
    case low = "low"
    case moderate = "moderate"
    case high = "high"
    case veryHigh = "veryHigh"
    
    var displayName: String {
        switch self {
        case .low: return "Low Risk"
        case .moderate: return "Moderate Risk"
        case .high: return "High Risk"
        case .veryHigh: return "Very High Risk"
        }
    }
    
    var color: String {
        switch self {
        case .low: return "green"
        case .moderate: return "yellow"
        case .high: return "orange"
        case .veryHigh: return "red"
        }
    }
}

// MARK: - Default Survey Data
extension SurveyData {
    static var defaultData: SurveyData {
        return SurveyData()
    }
    
    static var sampleData: SurveyData {
        return SurveyData(
            smokingStatus: .never,
            cigarettesPerDay: 0,
            vapingStatus: .never,
            puffsPerDay: 0,
            alcoholUnitsPerWeek: 5.0,
            fastFoodMealsPerWeek: 2,
            exerciseFrequency: .regular,
            isCompleted: true,
            lastUpdated: Date()
        )
    }
}